# Project Management API

Complete bilingual (English/Arabic) project management system with image uploads.

## Features

- ✅ Bilingual support (English and Arabic) for all text fields
- ✅ Image uploads (up to 10 images per project, 10MB each)
- ✅ Auto-generated sequential numbers (Sr. No)
- ✅ Draft/Published status
- ✅ Featured on homepage toggle
- ✅ Full CRUD operations
- ✅ Pagination and filtering
- ✅ Image management (add/remove)

## API Endpoints

### Public Routes

#### Get All Projects
```http
GET /api/projects
```

**Query Parameters:**
- `status` - Filter by status (draft/published)
- `featuredOnHomepage` - Filter featured projects (true/false)
- `page` - Page number (default: 1)
- `limit` - Items per page (default: 10)
- `sortBy` - Field to sort by (default: createdAt)
- `sortOrder` - Sort order (asc/desc, default: desc)

**Example:**
```http
GET /api/projects?status=published&featuredOnHomepage=true&page=1&limit=10
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "projects": [
      {
        "_id": "...",
        "srNo": 1,
        "title": {
          "en": "Luxury Villa Interior",
          "ar": "ديكورات داخلية فاخرة للفيلا"
        },
        "description": {
          "en": "Elegant marble flooring...",
          "ar": "تضفي الأرضيات الرخامية..."
        },
        "images": [
          {
            "url": "/uploads/projects/1234567890-image.jpg",
            "filename": "1234567890-image.jpg"
          }
        ],
        "featuredOnHomepage": true,
        "status": "published",
        "createdBy": {...},
        "createdAt": "2024-01-01T00:00:00.000Z",
        "updatedAt": "2024-01-01T00:00:00.000Z"
      }
    ],
    "pagination": {
      "page": 1,
      "limit": 10,
      "total": 25,
      "pages": 3
    }
  }
}
```

#### Get Project by ID
```http
GET /api/projects/:id
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "project": {...}
  }
}
```

### Protected Routes (Require Authentication)

#### Create Project
```http
POST /api/projects
Content-Type: multipart/form-data
Authorization: Bearer <token>
```

**Form Data:**
- `title[en]` - English title (required, max 300 chars)
- `title[ar]` - Arabic title (required, max 300 chars)
- `description[en]` - English description (required, max 5000 chars)
- `description[ar]` - Arabic description (required, max 5000 chars)
- `images` - Image files (optional, up to 10 files, 10MB each, PNG/JPG/WEBP)
- `featuredOnHomepage` - Boolean (optional, default: false)
- `status` - String (optional, draft/published, default: draft)

**Example using FormData:**
```javascript
const formData = new FormData();
formData.append('title[en]', 'Luxury Villa Interior');
formData.append('title[ar]', 'ديكورات داخلية فاخرة للفيلا');
formData.append('description[en]', 'Elegant marble flooring...');
formData.append('description[ar]', 'تضفي الأرضيات الرخامية...');
formData.append('featuredOnHomepage', 'true');
formData.append('status', 'published');
formData.append('images', file1);
formData.append('images', file2);
```

**Success Response (201):**
```json
{
  "success": true,
  "message": "Project created successfully",
  "data": {
    "project": {...}
  }
}
```

#### Update Project
```http
PUT /api/projects/:id
Content-Type: multipart/form-data
Authorization: Bearer <token>
```

**Form Data:**
- Same as create, all fields optional
- `removeImages` - Array of filenames to remove (optional)

**Example:**
```javascript
const formData = new FormData();
formData.append('title[en]', 'Updated Title');
formData.append('removeImages', 'old-image.jpg');
formData.append('images', newFile);
```

**Success Response (200):**
```json
{
  "success": true,
  "message": "Project updated successfully",
  "data": {
    "project": {...}
  }
}
```

#### Delete Project
```http
DELETE /api/projects/:id
Authorization: Bearer <token>
```

**Success Response (200):**
```json
{
  "success": true,
  "message": "Project deleted successfully"
}
```

#### Toggle Featured Status
```http
PATCH /api/projects/:id/feature
Authorization: Bearer <token>
```

**Success Response (200):**
```json
{
  "success": true,
  "message": "Project featured successfully",
  "data": {
    "project": {...}
  }
}
```

## Project Model Schema

```javascript
{
  srNo: Number (auto-generated, unique, sequential)
  title: {
    en: String (required, max 300 chars)
    ar: String (required, max 300 chars)
  }
  description: {
    en: String (required, max 5000 chars)
    ar: String (required, max 5000 chars)
  }
  images: [{
    url: String (file URL)
    filename: String (file name)
  }] (max 10 images)
  featuredOnHomepage: Boolean (default: false)
  status: String (enum: ['draft', 'published'], default: 'draft')
  createdBy: ObjectId (ref: Admin)
  createdAt: Date (auto)
  updatedAt: Date (auto)
}
```

## File Upload Specifications

- **Max Files:** 10 images per project
- **Max File Size:** 10MB per file
- **Allowed Types:** JPEG, JPG, PNG, WEBP
- **Storage:** `uploads/projects/` directory
- **URL Format:** `/uploads/projects/{filename}`

## Validation Rules

### Title
- Both English and Arabic required
- Maximum 300 characters each
- Trimmed automatically

### Description
- Both English and Arabic required
- Maximum 5000 characters each
- Trimmed automatically

### Images
- Maximum 10 images per project
- Each file max 10MB
- Only image files allowed (JPEG, JPG, PNG, WEBP)

## Error Responses

### 400 Bad Request
```json
{
  "success": false,
  "error": "Title in both English and Arabic is required"
}
```

### 401 Unauthorized
```json
{
  "success": false,
  "error": "Not authorized to access this route"
}
```

### 404 Not Found
```json
{
  "success": false,
  "error": "Project not found"
}
```

## Usage Examples

### Frontend Integration

#### Create Project with Images
```javascript
const createProject = async (projectData, imageFiles) => {
  const formData = new FormData();
  
  formData.append('title[en]', projectData.titleEn);
  formData.append('title[ar]', projectData.titleAr);
  formData.append('description[en]', projectData.descriptionEn);
  formData.append('description[ar]', projectData.descriptionAr);
  formData.append('featuredOnHomepage', projectData.featuredOnHomepage);
  formData.append('status', projectData.status);
  
  // Append images
  imageFiles.forEach(file => {
    formData.append('images', file);
  });

  const response = await fetch('http://localhost:3000/api/projects', {
    method: 'POST',
    headers: {
      'Authorization': `Bearer ${token}`
    },
    body: formData
  });

  return await response.json();
};
```

#### Update Project
```javascript
const updateProject = async (projectId, updates, newImages, imagesToRemove) => {
  const formData = new FormData();
  
  if (updates.titleEn) formData.append('title[en]', updates.titleEn);
  if (updates.titleAr) formData.append('title[ar]', updates.titleAr);
  if (updates.descriptionEn) formData.append('description[en]', updates.descriptionEn);
  if (updates.descriptionAr) formData.append('description[ar]', updates.descriptionAr);
  
  // Add new images
  newImages.forEach(file => {
    formData.append('images', file);
  });
  
  // Remove old images
  imagesToRemove.forEach(filename => {
    formData.append('removeImages', filename);
  });

  const response = await fetch(`http://localhost:3000/api/projects/${projectId}`, {
    method: 'PUT',
    headers: {
      'Authorization': `Bearer ${token}`
    },
    body: formData
  });

  return await response.json();
};
```

## Notes

- Sequential numbers (srNo) are auto-generated and cannot be edited
- Images are automatically deleted when a project is deleted
- Featured projects can be filtered using `featuredOnHomepage=true` query parameter
- Draft projects are not visible in public listings unless explicitly queried
- All text fields support both English and Arabic content
- Character limits are enforced: 300 for titles, 5000 for descriptions

