# Database Connection Troubleshooting Guide

## Common Issues and Solutions

### 1. Missing .env File

**Problem:** `.env` file doesn't exist, so `MONGODB_URI` is not set.

**Solution:**
```bash
# Copy .env.example to .env
cp .env.example .env

# Or on Windows PowerShell:
Copy-Item .env.example .env
```

Then edit `.env` and set your MongoDB connection string:
```env
MONGODB_URI=mongodb://localhost:27017/samaa
```

### 2. MongoDB Not Running

**Problem:** MongoDB service is not running on your system.

**Check if MongoDB is running:**
```bash
# Windows
net start MongoDB

# Or check if MongoDB service exists
Get-Service MongoDB

# Linux/Mac
sudo systemctl status mongod
# or
brew services list | grep mongodb
```

**Start MongoDB:**
```bash
# Windows
net start MongoDB

# Linux
sudo systemctl start mongod

# Mac
brew services start mongodb-community
```

### 3. Wrong Connection String

**Problem:** Connection string format is incorrect.

**Correct formats:**
```env
# Local MongoDB (default)
MONGODB_URI=mongodb://localhost:27017/samaa

# With authentication
MONGODB_URI=mongodb://username:password@localhost:27017/samaa

# MongoDB Atlas (cloud)
MONGODB_URI=mongodb+srv://username:password@cluster.mongodb.net/samaa

# Using 127.0.0.1 instead of localhost
MONGODB_URI=mongodb://127.0.0.1:27017/samaa
```

### 4. Connection Timeout

**Problem:** MongoDB is running but connection times out.

**Check:**
1. MongoDB is listening on port 27017:
   ```bash
   # Windows
   netstat -an | findstr 27017
   
   # Linux/Mac
   netstat -an | grep 27017
   ```

2. Firewall is not blocking the connection
3. Increase timeout in `.env`:
   ```env
   DB_CONNECT_TIMEOUT=30000
   DB_SERVER_SELECTION_TIMEOUT=10000
   ```

### 5. MongoDB Not Installed

**Problem:** MongoDB is not installed on your system.

**Install MongoDB:**
- **Windows:** Download from https://www.mongodb.com/try/download/community
- **Mac:** `brew install mongodb-community`
- **Linux:** Follow instructions at https://docs.mongodb.com/manual/installation/

### 6. Check Server Logs

When you start the server, check the logs for connection errors:

```bash
npm run dev
```

Look for messages like:
- ✅ `Database connection initialized` - Success
- ⚠️ `MONGODB_URI not set` - Missing .env file
- ❌ `MongoDB connection error` - Connection failed
- ❌ `Failed to connect to MongoDB` - Connection failed after retries

### 7. Test Connection Manually

Test MongoDB connection using MongoDB shell:
```bash
# Connect to MongoDB
mongosh

# Or older versions
mongo

# Test connection
show dbs
```

### 8. Verify Environment Variables

Check if environment variables are loaded:
```bash
# In Node.js, add this temporarily to server.js
console.log('MONGODB_URI:', process.env.MONGODB_URI);
```

### 9. Default Fallback

The code has a default fallback:
```javascript
uri: process.env.MONGODB_URI || "mongodb://localhost:27017/samaa"
```

So even without `.env`, it should try to connect to `localhost:27017`. If this fails, MongoDB is likely not running.

## Quick Fix Checklist

1. ✅ Create `.env` file from `.env.example`
2. ✅ Set `MONGODB_URI` in `.env`
3. ✅ Ensure MongoDB is installed
4. ✅ Ensure MongoDB service is running
5. ✅ Check MongoDB is listening on port 27017
6. ✅ Restart the Node.js server
7. ✅ Check server logs for connection status

## Health Check Endpoint

After starting the server, check the health endpoint:
```bash
GET http://localhost:3000/health
```

Response will show database connection status:
```json
{
  "status": "healthy",
  "database": {
    "status": "healthy",
    "state": "connected",
    "isConnected": true,
    "host": "localhost",
    "port": 27017,
    "database": "samaa"
  }
}
```

## Still Having Issues?

1. Check MongoDB logs (usually in MongoDB installation directory)
2. Try connecting with MongoDB Compass GUI
3. Verify network connectivity
4. Check if another application is using port 27017
5. Try using `127.0.0.1` instead of `localhost`

