/**
 * Validation utilities
 */

/**
 * Validate MongoDB ObjectId
 */
export const isValidObjectId = (id) => {
  return /^[0-9a-fA-F]{24}$/.test(id);
};

/**
 * Validate language parameter
 */
export const isValidLanguage = (lang) => {
  return lang === "en" || lang === "ar" || !lang;
};

/**
 * Validate pagination parameters
 */
export const validatePagination = (page, limit) => {
  const pageNum = parseInt(page) || 1;
  const limitNum = parseInt(limit) || 10;

  return {
    page: Math.max(1, pageNum),
    limit: Math.min(Math.max(1, limitNum), 100), // Max 100 items per page
    skip: (Math.max(1, pageNum) - 1) * Math.min(Math.max(1, limitNum), 100),
  };
};

/**
 * Validate project status
 */
export const isValidStatus = (status) => {
  return status === "draft" || status === "published";
};

/**
 * Sanitize string input
 */
export const sanitizeString = (str) => {
  if (typeof str !== "string") return "";
  return str.trim().replace(/\s+/g, " ");
};

/**
 * Escape special regex characters for safe search
 */
export const escapeRegex = (str) => {
  if (typeof str !== "string") return "";
  return str.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
};

/**
 * Validate search query
 */
export const validateSearchQuery = (search) => {
  if (!search || typeof search !== "string") {
    return { valid: false, error: "Search query must be a non-empty string" };
  }

  const trimmed = search.trim();
  if (trimmed.length === 0) {
    return { valid: false, error: "Search query cannot be empty" };
  }

  if (trimmed.length > 100) {
    return { valid: false, error: "Search query cannot exceed 100 characters" };
  }

  return { valid: true, search: trimmed };
};

/**
 * Validate bilingual title
 */
export const validateBilingualTitle = (title) => {
  if (!title || typeof title !== "object") {
    return {
      valid: false,
      error: "Title must be an object with 'en' and 'ar' properties",
    };
  }

  if (!title.en || typeof title.en !== "string" || !title.en.trim()) {
    return { valid: false, error: "English title is required" };
  }

  if (!title.ar || typeof title.ar !== "string" || !title.ar.trim()) {
    return { valid: false, error: "Arabic title is required" };
  }

  if (title.en.trim().length > 300) {
    return {
      valid: false,
      error: "English title cannot exceed 300 characters",
    };
  }

  if (title.ar.trim().length > 300) {
    return { valid: false, error: "Arabic title cannot exceed 300 characters" };
  }

  return { valid: true };
};

/**
 * Validate bilingual description
 */
export const validateBilingualDescription = (description) => {
  if (!description || typeof description !== "object") {
    return {
      valid: false,
      error: "Description must be an object with 'en' and 'ar' properties",
    };
  }

  if (
    !description.en ||
    typeof description.en !== "string" ||
    !description.en.trim()
  ) {
    return { valid: false, error: "English description is required" };
  }

  if (
    !description.ar ||
    typeof description.ar !== "string" ||
    !description.ar.trim()
  ) {
    return { valid: false, error: "Arabic description is required" };
  }

  if (description.en.trim().length > 5000) {
    return {
      valid: false,
      error: "English description cannot exceed 5000 characters",
    };
  }

  if (description.ar.trim().length > 5000) {
    return {
      valid: false,
      error: "Arabic description cannot exceed 5000 characters",
    };
  }

  return { valid: true };
};
